/*
    REMOVE_DISPLAY_CHANNEL is a program that run under PMDF to remove
    the "display name" from the rfc2822.from header of an Internet email
    message.
    Copyright (C) 2020 Francesco Gennai

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

    Francesco Gennai - francesco.gennai@gmail.com
                       francesco.gennai@isti.cnr.it
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef __VMS
#include "pmdf_com:apidef.h"
#else
#include "/pmdf/include/apidef.h"
#endif

#include "remove_display.h"

#define FG__VERSION "1.1"

typedef char string[FG__STRINGLEN];
typedef char bigstring[FG__BIGSTRINGLEN];

string from_adr;
int from_adr_len;
PMDF_nq *nq_context = 0;
PMDF_dq *dq_context = 0;
PMDF_opt *option_ctx = 0;

void check (int stat, char *field, int *retstatus);
int get_message ();
int load_options (string channel);
int selectheaderFrom(PMDF_hdr *hdr, char *original_from, char *new_from);

int main () {

  string channel, env_id, orig_adr, to_adr, message_file, errortext, 
         original_from;
  bigstring headerline, new_from;
  int channel_len, env_id_len, nflags, i, orig_adr_len, to_adr_len, 
      disposition, retstatus, errorlen, FG__debuglevel;
  char *dispositiontext, *field;
  PMDF_hdr *hdr;
  unsigned int key;


  retstatus = PMDFinitialize (1);
     if (retstatus != PMDF__OK) {  
          FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFinitialize failure, %d",
                  __func__, retstatus);
          return (retstatus);
     }

  channel_len = ALFA_SIZE;
  retstatus = PMDFgetChannelName (channel, &channel_len, &key, &key);
     if (retstatus != PMDF__OK) {  
          FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFgetChannelName failure, %d",
                  __func__, retstatus);
          return (retstatus);
     }

  retstatus = load_options(channel);

     if (!(retstatus)) {  
          FG__DIAGNOSTIC(FG__ERROR, "(%s) load_options routine failure, %d",
                  __func__, retstatus);
          return (retstatus);
     }


  PMDFdequeueInitialize (&dq_context);

  PMDFenqueueInitialize ();

  /* Loop to get messages from the queue */
  while (get_message ()) {
    env_id_len = ALFA_SIZE;

    field = "PMDFgetEnvelopeId";
    check (PMDFgetEnvelopeId (&dq_context, env_id,&env_id_len),field,&retstatus);
    if (retstatus != PMDF__OK) continue;

    FG__DIAGNOSTIC(FG__NORMAL, "(%s) Message envelope id: %s",
                          __func__, env_id);

    retstatus = PMDFstartMessageEnvelope (&nq_context, channel, channel_len,
                                     from_adr, from_adr_len);
       if (retstatus != PMDF__OK) {  
            if (retstatus == PMDF__NO) {
               memset(errortext, 0, sizeof(*errortext) * FG__STRINGLEN); 
               PMDFgetErrorText(&nq_context, errortext, &errorlen);
               FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFstartMessageEnvelope failure, PMDF errtxt: %s %d",
                    __func__, errortext, retstatus);
            }

            field = "PMDFstartMessageEnvelope";
            check(retstatus, field, &retstatus);
            continue;
       }

    field = "PMDFsetEnvelopeId";
    check (PMDFsetEnvelopeId (&nq_context, env_id, env_id_len),field,&retstatus);
    if (retstatus != PMDF__OK) continue;

    to_adr_len = orig_adr_len = ALFA_SIZE;

    /* Loop: get envelope to addresses from the incoming message, write 
             envelope to addresses to the outgoing message */
    while (1 & PMDFgetRecipient (&dq_context, to_adr, &to_adr_len,
                                     orig_adr, &orig_adr_len))

    {

      PMDFgetRecipientFlags (&dq_context, &nflags);
      PMDFsetRecipientFlags (&nq_context, nflags);
      retstatus = PMDFaddRecipient (&nq_context, to_adr, to_adr_len,
                               orig_adr, orig_adr_len);
      errorlen = ALFA_SIZE; 
      memset(errortext, 0, sizeof(*errortext) * FG__STRINGLEN); 
      disposition = PMDF_DISP_DEFERRED;
      dispositiontext = "PMDF_DISP_DEFERRED";
      FG__debuglevel = FG__ERROR;

      if ( (retstatus == PMDF__HOST) || (retstatus == PMDF__USER) ||
            (retstatus == PMDF__NAUTH) || (retstatus == PMDF__PARSE) ) {

          disposition = PMDF_DISP_FAILED;
          dispositiontext = "PMDF_DISP_FAILED";
          PMDFgetErrorText(&nq_context, errortext, &errorlen);
          FG__debuglevel = FG__NORMAL;
      }
      else if (retstatus == PMDF__OK) { 

          disposition = PMDF_DISP_DELIVERED;
          dispositiontext = "PMDF_DISP_DELIVERED";
          FG__debuglevel = FG__NORMAL;

      }

      FG__DIAGNOSTIC(FG__debuglevel, "(%s) Recipient - to_adr: %s - orig_adr: %s",
                      __func__, to_adr, orig_adr);
      FG__DIAGNOSTIC(FG__debuglevel, "(%s) addr disposition: %s %s",
                      __func__, dispositiontext, to_adr);
      if (strlen(errortext) != 0) 
                   FG__DIAGNOSTIC(FG__debuglevel, "(%s) PMDF errtxt: %s %s",
                    __func__, errortext, to_adr);

      retstatus = PMDFrecipientDisposition (&dq_context, nflags,
                                       disposition, to_adr,
                                       to_adr_len, orig_adr, orig_adr_len,
                                       errortext, strlen(errortext));
      if (retstatus != PMDF__OK) {  
           FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFrecipientDisposition failure, %d",
                   __func__, retstatus);
           FG__DIAGNOSTIC(FG__ERROR, "(%s) Disposition not set %s",
                   __func__, to_adr);
           continue;
      }

      to_adr_len = orig_adr_len = ALFA_SIZE;
    } 

    field = "PMDFstartMessageHeader";
    check (PMDFstartMessageHeader (&nq_context),field,&retstatus);
    if (retstatus != PMDF__OK) continue;
    
    field = "PMDFreadHeader";
    check (PMDFreadHeader (&dq_context, &hdr),field,&retstatus);
    if (retstatus != PMDF__OK) continue;

    retstatus = selectheaderFrom(hdr, original_from, new_from);
    if (retstatus) {
       retstatus = PMDFdeleteHeaderLine (hdr, HL_FROM);
       if (retstatus == PMDF__OK) {
           FG__DIAGNOSTIC(FG__NORMAL, "(%s) original header From: deleted",
                          __func__);
           retstatus = PMDFwriteFrom (&nq_context, new_from, strlen(new_from));
           if (retstatus != PMDF__OK) {
              FG__DIAGNOSTIC(FG__NORMAL, "(%s) error adding new header From:",
                             __func__); 
              /* try to rewrite the original from */
              PMDFwriteFrom (&nq_context, original_from, strlen(original_from));
           } else {
              FG__DIAGNOSTIC(FG__NORMAL, "(%s) header added, From: %s",
                          __func__, new_from);
              snprintf(headerline, FG__BIGSTRINGLEN, "X-original-from: %s", 
                       original_from);
              PMDFwriteLine (&nq_context, headerline, strlen(headerline));
              FG__DIAGNOSTIC(FG__NORMAL, "(%s) header added, %s",
                          __func__, headerline);
           }
       }
    }

    field = "PMDFwriteHeader";
    check (PMDFwriteHeader (&nq_context, hdr),field,&retstatus);
    if (retstatus != PMDF__OK) {
        if (hdr) PMDFdisposeHeader (&hdr); 
        continue;
    } else {
       FG__DIAGNOSTIC(FG__NORMAL, "(%s) %s routine OK",
                          __func__, field);
    }

    PMDFdisposeHeader (&hdr);

    field = "PMDFstartMessageBody";
    check (PMDFstartMessageBody (&nq_context),field,&retstatus);
    if (retstatus != PMDF__OK) 
       continue;
    else
       FG__DIAGNOSTIC(FG__NORMAL, "(%s) %s routine OK",
                          __func__, field);

    field = "PMDFcopyMessage";
    check (PMDFcopyMessage (&dq_context, &nq_context),field,&retstatus);
    if (retstatus != PMDF__OK)
       continue;
    else
       FG__DIAGNOSTIC(FG__NORMAL, "(%s) %s routine OK",
                          __func__, field);

    field = "PMDFenqueueMessage";
    retstatus = PMDFenqueueMessage (&nq_context);
    if (retstatus != PMDF__OK) {
       if (retstatus != PMDF__NOOP) {
           memset(errortext, 0, sizeof(*errortext) * FG__STRINGLEN); 
           PMDFgetErrorText(&nq_context, errortext, &errorlen);
           FG__DIAGNOSTIC(FG__ERROR, 
                  "(%s) %s failure, PMDF errtxt: %s %d",
                   __func__, field, errortext, retstatus);
           check(retstatus, field, &retstatus);
           continue;
       } else {
           FG__DIAGNOSTIC(FG__ERROR,
                  "(%s) %s: message had no envelope TO: address; message has been deleted", 
                  __func__, field);
       }
    } else {
           FG__DIAGNOSTIC(FG__NORMAL, "(%s) %s routine OK",
                          __func__, field);
    }

    field = "PMDFdequeueMessageEnd";
    retstatus = PMDFdequeueMessageEnd (&dq_context, 0, NULL, 0);
    if (retstatus == PMDF__OK)
       FG__DIAGNOSTIC(FG__NORMAL, "(%s) %s routine OK",
                          __func__, field);

  }

  PMDFdequeueEnd (&dq_context);
  PMDFdone ();
}


/***********************************/
/***********************************/

void check (int stat, char *field, int *retstatus)
{

  string reason;
  *retstatus = stat;
  if (stat != PMDF__OK) {  
       FG__DIAGNOSTIC(FG__ERROR, "(%s) %s failure, %d",
                    __func__, field, stat);
           if (dq_context) {
             snprintf(reason,ALFA_SIZE,"Error in routine %s - error code %d",
                      field,stat);
             FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFdequeueMessageEnd",__func__);
             PMDFdequeueMessageEnd (&dq_context, 1, reason, strlen(reason));
           }
           FG__DIAGNOSTIC(FG__ERROR, "(%s) Message deferred",
                    __func__);
           if (nq_context) {
             FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFabortMessage",__func__);
             PMDFabortMessage (&nq_context);
           }
  }

}


int get_message ()
{
    string msg_file;
    int msg_file_len, stat;

    memset(msg_file, 0, sizeof(*msg_file) * FG__STRINGLEN); 
    memset(from_adr, 0, sizeof(*from_adr) * FG__STRINGLEN); 

    msg_file_len = from_adr_len = ALFA_SIZE;
    stat = PMDFgetMessage (&dq_context, msg_file, &msg_file_len,
                           from_adr, &from_adr_len);

    if (!(1 & stat) && stat != PMDF__EOF)
           FG__DIAGNOSTIC(FG__ERROR, "(%s) PMDFgetMessage failure, file: %s, %d",
                  __func__, msg_file, stat);

    if (stat == PMDF__OK)
        FG__DIAGNOSTIC(FG__NORMAL, "(%s) PMDFgetMessage, envelope from: %s, file: %s",
                    __func__, from_adr, msg_file);

    if (stat == PMDF__EOF)
        FG__DIAGNOSTIC(FG__NORMAL, "(%s) PMDFgetMessage, NO MORE MESSAGES TO BE PROCESSED",
                    __func__);

    return (1 & stat);
}


int load_options (string channel) {

    int option_len, stat;
    string option_file,  in_bckdomain;
    char *bckdomainP;

    snprintf (option_file, ALFA_SIZE+1, "/pmdf_table/%s_option.", channel);

    FG__debug = FG__DEBUG_DEFAULT;

    FG__DIAGNOSTIC(FG__NORMAL, "(%s) option file: %s",
                        __func__, option_file);

    stat = PMDFoptionRead (&option_ctx, option_file,
                                   strlen (option_file));
    if (stat != PMDF__OK) return (1 & stat);


    if (option_ctx) {

         stat = PMDFoptionGetInteger (option_ctx,
                                    FG__OPTION_DEBUG,
                                    strlen (FG__OPTION_DEBUG), &FG__debug);
         if (stat != PMDF__OK) return (1 & stat);

         FG__DIAGNOSTIC(FG__ERROR, "(%s) REMDISPLAY channel program version: %s",
               __func__, FG__VERSION);
        
         FG__DIAGNOSTIC(FG__NORMAL, "(%s) debug - %s = %d",
                          __func__, FG__OPTION_DEBUG, FG__debug);

    }

    stat = PMDFoptionDispose (option_ctx);
    return (1 & stat);
}


int selectheaderFrom(PMDF_hdr *hdr, char *original_from, char *new_from) {
 string localpart, domainpart, phrase;
 PMDF_addr *addr_context = 0;
 PMDF_hdr_line *hdr_line;
 int iaddr, count, addrlen=ALFA_SIZE+1, from_offset = 5, retstatus = 0,
     localpartlen=FG__STRINGLEN, domainpartlen=FG__STRINGLEN, 
     phraselen=FG__STRINGLEN;


    hdr_line = (*hdr)[HL_FROM];


    if (hdr_line)  {

      FG__DIAGNOSTIC(FG__NORMAL, "(%s) hdr_line = %s",
                        __func__, hdr_line->line);

      snprintf(original_from,ALFA_SIZE,"%s",&hdr_line->line[from_offset]);

      PMDFaddressParseList(&addr_context, &count, &hdr_line->line[from_offset],
                           strlen(&hdr_line->line[from_offset]) );
      if (count) {

        iaddr = 1;

        PMDFaddressGetProperty(addr_context, iaddr, PMDF_PROP_PHRASE,
                               phrase, &phraselen);
        if (phraselen == 0) {

           FG__DIAGNOSTIC(FG__NORMAL, "(%s) no display name: DO NOTHING",
                          __func__);
        } else {

           FG__DIAGNOSTIC(FG__NORMAL, "(%s) Display name: %s",
                          __func__, phrase);

           PMDFaddressGetProperty(addr_context, iaddr, PMDF_PROP_LOCAL,
                               localpart, &localpartlen);
           PMDFaddressGetProperty(addr_context, iaddr, PMDF_PROP_DOMAIN,
                               domainpart, &domainpartlen);
       
           snprintf(new_from,FG__BIGSTRINGLEN,"%s@%s",localpart,domainpart);

           FG__DIAGNOSTIC(FG__NORMAL, "(%s) new header from: %s",
                          __func__, new_from);
           retstatus = 1;
        }

      }

      PMDFaddressDispose(addr_context);
    }

    return (retstatus);
}

